#include "capy.h"
#ifndef FIXTURE
#define FIXTURE

// Test function
static void fun(void) {
  raiseExc(CapyExc_MallocFailed);
}

// Test method
typedef struct TestMethod {
  int a, b, c;
  CapyPad(int[3], 0);
  int (*get)(void);
} TestMethod;

static int TestMethodGet(void) {
  methodOf(TestMethod);
  that->b = 2;
  if(that->a == 1) raiseExc(CapyExc_MallocFailed);
  that->c = 3;
  return that->a;
}

// User define exceptions
typedef enum UserDefinedExceptions {
  conflictException = CapyExc_LastID - 1,
  myUserExceptionA = CapyExc_LastID,
} UserDefinedExceptions;

char const* exceptionStr[] = {
  "conflictException",
  "myUserExceptionA",
};

static char const* ExcToStr(
  CapyException_t exc) {
  if(
    exc >= conflictException &&
    exc <= myUserExceptionA
  ) {
    return exceptionStr[exc - conflictException];
  } else {
    return NULL;
  }
}

#define NB_FLAGS 10
struct {bool flags[NB_FLAGS];} volatileVar;
#endif
CUTEST(test001, "Raise an exception in a TryCatch block and catch it") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_MallocFailed);
    volatileVar.flags[0] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[1] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1],
    "unexpected exception handling");
  CUTEST_ASSERT(
    CapyGetLastExcId() == CapyExc_MallocFailed,
    "unexpected exception handling");
}

CUTEST(test002, "Raise an exception in a TryCatch block and ignore it") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_MallocFailed);
    volatileVar.flags[0] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0],
    "unexpected exception handling");
}

CUTEST(
  test003,
  "TryCatch block inside another TryCatch block, exception ignored by the "
  "inner block, hence not seen by the outer block") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    try {
      raiseExc(CapyExc_MallocFailed);
      volatileVar.flags[0] = true;
    } endCatch;
    volatileVar.flags[1] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[2] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1] &&
    !volatileVar.flags[2],
    "unexpected exception handling");
}

CUTEST(
  test004,
  "TryCatch block inside another TryCatch block, exception forwarded by "
  "the inner block, caught by the outer block") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    try {
      raiseExc(CapyExc_MallocFailed);
      volatileVar.flags[0] = true;
    } endCatch;
    CapyForwardExc();
    volatileVar.flags[1] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[2] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && !volatileVar.flags[1] &&
    volatileVar.flags[2],
    "unexpected exception handling");
}

CUTEST(test005, "Forward when there is no exception") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    volatileVar.flags[0] = true;
  } endCatch;
  CapyForwardExc();
  volatileVar.flags[1] = true;
  CUTEST_ASSERT(
    volatileVar.flags[0] && volatileVar.flags[1],
    "unexpected exception handling");
}

CUTEST(
  test006,
  "TryCatch block inside another TryCatch block, exception caught and "
  "forwarded by the inner block, caught by the outer block") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    try {
      raiseExc(CapyExc_MallocFailed);
      volatileVar.flags[0] = true;
    } catch(CapyExc_MallocFailed) {
      volatileVar.flags[1] = true;
    } endCatch;
    CapyForwardExc();
    volatileVar.flags[2] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[3] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1] &&
    !volatileVar.flags[2] && volatileVar.flags[3],
    "unexpected exception handling");
}

CUTEST(
  test007,
  "TryCatch block inside another TryCatch block, exception caught and "
  "cancelled by the inner block, hence unseen by the outer block") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    try {
      raiseExc(CapyExc_MallocFailed);
      volatileVar.flags[0] = true;
    } catch(CapyExc_MallocFailed) {
      volatileVar.flags[1] = true;
      CapyCancelExc();
    } endCatch;
    CapyForwardExc();
    volatileVar.flags[2] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[3] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1] &&
    volatileVar.flags[2] && !volatileVar.flags[3],
    "unexpected exception handling");
}

CUTEST(
  test008,
  "Raise exception in TryCacth block with mulitple catch segments") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_MallocFailed);
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[0] = true;
  } catch(CapyExc_StreamReadError) {
    volatileVar.flags[1] = true;
  } endCatch;
  CUTEST_ASSERT(
    volatileVar.flags[0] && !volatileVar.flags[1],
    "unexpected exception handling");
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_MallocFailed);
  } catch(CapyExc_StreamReadError) {
    volatileVar.flags[0] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[1] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1],
    "unexpected exception handling");
}

CUTEST(
  test009,
  "Raise exception in TryCacth block with shared catch segments") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_MallocFailed);
  } catch(CapyExc_MallocFailed)
  catchAlso(CapyExc_StreamReadError) {
    volatileVar.flags[0] = true;
  } endCatch;
  CUTEST_ASSERT(
    volatileVar.flags[0],
    "unexpected exception handling");
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_StreamReadError);
  } catch(CapyExc_MallocFailed)
  catchAlso(CapyExc_StreamReadError) {
    volatileVar.flags[0] = true;
  } endCatch;
  CUTEST_ASSERT(
    volatileVar.flags[0],
    "unexpected exception handling");
}

CUTEST(
  test010,
  "Raise exception in TryCacth block with default catch segment") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_MallocFailed);
  } catchDefault {
    volatileVar.flags[0] = true;
  } endCatch;
  CUTEST_ASSERT(
    volatileVar.flags[0],
    "unexpected exception handling");
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_StreamReadError);
  } catchDefault {
    volatileVar.flags[0] = true;
  } endCatch;
  CUTEST_ASSERT(
    volatileVar.flags[0],
    "unexpected exception handling");
}

CUTEST(
  test011,
  "Raised exception in called function, caught by the TryCatch block of "
  "the calling function") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    fun();
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[0] = true;
  } endCatch;
  CUTEST_ASSERT(
    volatileVar.flags[0],
    "unexpected exception handling");
}

CUTEST(
  test012,
  "Raised exception from a catch block, recaught by the same TryCatch block") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(CapyExc_StreamReadError);
    volatileVar.flags[0] = true;
  } catch(CapyExc_StreamReadError) {
    volatileVar.flags[1] = true;
    raiseExc(CapyExc_MallocFailed);
    volatileVar.flags[2] = true;
  } catch(CapyExc_MallocFailed) {
    volatileVar.flags[3] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1] &&
    !volatileVar.flags[2] && volatileVar.flags[3],
    "unexpected exception handling");
}

CUTEST(test013, "Examples with multithreading") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  #pragma omp parallel sections
  {
    #pragma omp section
    {
      try {
        volatileVar.flags[0] = true;
      } endCatch;
      volatileVar.flags[1] = true;
    }
    #pragma omp section
    {
      try {
        volatileVar.flags[2] = true;
      } endCatch;
      volatileVar.flags[3] = true;
    }
  }
  CUTEST_ASSERT(
    volatileVar.flags[0] && volatileVar.flags[1] &&
    volatileVar.flags[2] && volatileVar.flags[3],
    "unexpected exception handling");
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  #pragma omp parallel sections
  {
    #pragma omp section
    {
      try {
        volatileVar.flags[0] = true;
        raiseExc(CapyExc_MallocFailed);
        volatileVar.flags[1] = true;
      } catch(CapyExc_MallocFailed) {
        volatileVar.flags[2] = true;
      } endCatch;
      volatileVar.flags[3] = true;
    }
    #pragma omp section
    {
      try {
        volatileVar.flags[4] = true;
      } endCatch;
      volatileVar.flags[5] = true;
    }
  }
  CUTEST_ASSERT(
    volatileVar.flags[0] && !volatileVar.flags[1] &&
    volatileVar.flags[2] && volatileVar.flags[3] && volatileVar.flags[4] &&
    volatileVar.flags[5],
    "unexpected exception handling");
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    #pragma omp parallel sections
    {
      #pragma omp section
      {
        try {
          volatileVar.flags[0] = true;
          raiseExc(CapyExc_MallocFailed);
          volatileVar.flags[1] = true;
        } catch(CapyExc_MallocFailed) {
          volatileVar.flags[2] = true;
        } endCatch;
        volatileVar.flags[3] = true;
        CapyForwardExc();
      }
      #pragma omp section
      {
        try {
          volatileVar.flags[4] = true;
          raiseExc(CapyExc_StreamReadError);
          volatileVar.flags[5] = true;
        } catch(CapyExc_StreamReadError) {
          volatileVar.flags[6] = true;
        } endCatch;
        volatileVar.flags[7] = true;
        CapyForwardExc();
      }
    }
  } endCatch;
  CUTEST_ASSERT(
    CapyGetLastExcId() == 0 ||
    CapyGetLastExcId() == CapyExc_MallocFailed ||
    CapyGetLastExcId() == CapyExc_StreamReadError,
    "unexpected exception handling");
  CUTEST_ASSERT(
    volatileVar.flags[0] && !volatileVar.flags[1] &&
    volatileVar.flags[2] && volatileVar.flags[3] && volatileVar.flags[4] &&
    !volatileVar.flags[5] && volatileVar.flags[6] && volatileVar.flags[7],
    "unexpected exception handling");
}

CUTEST(
  test014,
  "Example of user-defined exception without setting the conversion "
  "function") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  try {
    raiseExc(myUserExceptionA);
    volatileVar.flags[0] = true;
  } catch(myUserExceptionA) {
    volatileVar.flags[1] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1],
    "unexpected exception handling");
}

CUTEST(
  test015,
  "Example of user-defined exception after setting the conversion "
  "function") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  CapyAddExcToStrFun(ExcToStr);
  try {
    raiseExc(myUserExceptionA);
    volatileVar.flags[0] = true;
  } catch(myUserExceptionA) {
    volatileVar.flags[1] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1],
    "unexpected exception handling");
}

CUTEST(
  test016,
  "Example of user-defined exception with conflicting ID and without "
  "conversion function") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  CapyAddExcToStrFun(ExcToStr);
  try {
    raiseExc(conflictException);
    volatileVar.flags[0] = true;
  } catch(conflictException) {
    volatileVar.flags[1] = true;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && volatileVar.flags[1],
    "unexpected exception handling");
}

CUTEST(
  test017,
  "Example of method raising exception") {
  loop(iFlag, NB_FLAGS) volatileVar.flags[iFlag] = false;
  TestMethod instance = {.a = 1, .b = 1, .c = 1, .get = TestMethodGet};
  try {
    volatileVar.flags[0] = $(&instance, get)();
    volatileVar.flags[1] = true;
  } catch (CapyExc_MallocFailed) {
    volatileVar.flags[2] = true;
    instance.c = 4;
  } endCatch;
  CUTEST_ASSERT(
    !volatileVar.flags[0] && !volatileVar.flags[1] &&
    volatileVar.flags[2] && instance.b == 2 && instance.c == 4,
    "unexpected exception handling");
}
