#include "capy.h"
#ifndef FIXTURE
#define FIXTURE
#endif
CUTEST(test001, "Alloc and free") {
  CapyTurtleGraphic* turtle = CapyTurtleGraphicAlloc();
  CUTEST_ASSERT(
    fabs(turtle->state.step.vals[0] - 1.0) < 1e-6 &&
    fabs(turtle->state.step.vals[1] - 0.0) < 1e-6 &&
    fabs(turtle->state.pos.vals[0] - 0.0) < 1e-6 &&
    fabs(turtle->state.pos.vals[1] - 0.0) < 1e-6,
    "turtle creation failed");
  CapyTurtleGraphicFree(&turtle);
}

CUTEST(test002, "Koch snowflake") {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_800x600);
  CapyLSysKochSnowflakeState init = {.val = 'F'};
  CapyLSysKochSnowflake* koch = CapyLSysKochSnowflakeAlloc(&init, 1);
  loop(i, 6) $(koch, step)();
  CapyTurtleGraphic* turtle = CapyTurtleGraphicAlloc();
  turtle->img = img;
  turtle->state.pos.vals[1] = 100;
  $(turtle, resize)(800.0);
  loop(i, koch->len) {
    switch(koch->state[i].val) {
      case 'D':
        $(turtle, resize)(1.0 / 3.0);
        break;
      case 'U':
        $(turtle, resize)(3.0);
        break;
      case 'F':
        $(turtle, forward)(1);
        break;
      case '+':
        $(turtle, turn)(M_PI / 3.0);
        break;
      case '-':
        $(turtle, turn)(-M_PI / 3.0);
        break;
      default:
        break;
    }
  }
  $(img, saveToPath)("UnitTests/TestTurtleGraphic/koch.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestTurtleGraphic/checkKoch.png");
  CUTEST_ASSERT($(img, isSame)(checkImg), "turtle on Koch snowflake failed");
  CapyTurtleGraphicFree(&turtle);
  CapyLSysKochSnowflakeFree(&koch);
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test003, "Fractal binary tree") {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_800x600);
  CapyLSysFractalBinaryTreeState init = {.val = '0'};
  CapyLSysFractalBinaryTree* tree = CapyLSysFractalBinaryTreeAlloc(&init, 1);
  loop(i, 6) $(tree, step)();
  CapyTurtleGraphic* turtle = CapyTurtleGraphicAlloc();
  turtle->img = img;
  turtle->state.pos.vals[0] = 400;
  turtle->state.pos.vals[1] = 600;
  turtle->state.step.vals[0] = 0;
  turtle->state.step.vals[1] = -10;
  loop(i, tree->len) {
    switch(tree->state[i].val) {
      case '0':
        $(turtle, forward)(1);
        break;
      case '1':
        $(turtle, forward)(1);
        break;
      case '[':
        $(turtle, push)();
        $(turtle, turn)(M_PI / 4.0);
        break;
      case ']':
        $(turtle, pop)();
        $(turtle, turn)(-M_PI / 4.0);
        break;
      default:
        break;
    }
  }
  $(img, saveToPath)("UnitTests/TestTurtleGraphic/tree.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestTurtleGraphic/checkTree.png");
  CUTEST_ASSERT(
    $(img, isSame)(checkImg), "turtle on fractal binary tree failed");
  CapyTurtleGraphicFree(&turtle);
  CapyLSysFractalBinaryTreeFree(&tree);
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test004, "Hilbert curve") {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_800x800);
  CapyLSysHilbertCurveState init = {.val = 'A'};
  CapyLSysHilbertCurve* hilbert = CapyLSysHilbertCurveAlloc(&init, 1);
  int n = 4;
  double l = 799.0 / 15.0;
  loop(i, n) {
    $(hilbert, step)();
  }
  CapyTurtleGraphic* turtle = CapyTurtleGraphicAlloc();
  turtle->img = img;
  $(turtle, resize)(l);
  loop(i, hilbert->len) {
    switch(hilbert->state[i].val) {
      case 'F':
        $(turtle, forward)(1);
        break;
      case '+':
        $(turtle, turn)(M_PI / 2.0);
        break;
      case '-':
        $(turtle, turn)(-M_PI / 2.0);
        break;
      default:
        break;
    }
  }
  $(img, saveToPath)("UnitTests/TestTurtleGraphic/hilbert.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestTurtleGraphic/checkHilbert.png");
  CUTEST_ASSERT($(img, isSame)(checkImg), "turtle on Hilbert failed");
  CapyTurtleGraphicFree(&turtle);
  CapyLSysHilbertCurveFree(&hilbert);
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test005, "Heighway dragon curve") {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_800x800);
  $(img, fillWithColor)(&capyColorRGBAWhite);
  CapyLSysHeighwayDragonCurveState inits[2] = {
    {.val = 'F'},
    {.val = 'X'},
  };
  CapyLSysHeighwayDragonCurve* dragon =
    CapyLSysHeighwayDragonCurveAlloc(inits, 2);
  int n = 12;
  double l = 5.0;
  loop(i, n) {
    $(dragon, step)();
  }
  CapyTurtleGraphic* turtle = CapyTurtleGraphicAlloc();
  turtle->img = img;
  double dirs[4][2] = {
    {1.0, 0.0}, {0.0, 1.0}, {-1.0, 0.0}, {0.0, -1.0},
  };
  CapyColorData colors[4] = {
    capyColorRGBABlack, capyColorRGBARed, capyColorRGBAGreen, capyColorRGBABlue,
  };
  loop(iRun, 4) {
    turtle->state.pos.vals[0] = 400.0;
    turtle->state.pos.vals[1] = 400.0;
    turtle->state.step.vals[0] = dirs[iRun][0];
    turtle->state.step.vals[1] = dirs[iRun][1];
    $(turtle, resize)(l);
    turtle->color = colors[iRun];
    loop(i, dragon->len) {
      switch(dragon->state[i].val) {
        case 'F':
          $(turtle, forward)(1);
          break;
        case '+':
          $(turtle, turn)(M_PI / 2.0);
          break;
        case '-':
          $(turtle, turn)(-M_PI / 2.0);
          break;
        default:
          break;
      }
    }
  }
  $(img, saveToPath)("UnitTests/TestTurtleGraphic/heighwayDragon.png");
  CapyImg* checkImg =
    CapyImgLoadFromPath("UnitTests/TestTurtleGraphic/checkHeighwayDragon.png");
  CUTEST_ASSERT($(img, isSame)(checkImg), "turtle on Heighway Dragon failed");
  CapyTurtleGraphicFree(&turtle);
  CapyLSysHeighwayDragonCurveFree(&dragon);
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}

CUTEST(test006, "Gosper curve") {
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, capyImgDims_800x800);
  $(img, fillWithColor)(&capyColorRGBAWhite);
  CapyLSysGosperCurveState inits[1] = {
    {.val = 'A'},
  };
  CapyLSysGosperCurve* curve = CapyLSysGosperCurveAlloc(inits, 1);
  int n = 4;
  double l = 400.0;
  loop(i, n) {
    $(curve, step)();
  }
  CapyTurtleGraphic* turtle = CapyTurtleGraphicAlloc();
  turtle->img = img;
  CapyColorData colors[3] = {
    capyColorRGBARed, capyColorRGBAGreen, capyColorRGBABlue,
  };
  loop(iRun, 3) {
    turtle->state.pos.vals[0] = 400.0;
    turtle->state.pos.vals[1] = 400.0;
    turtle->state.step.vals[0] = 1.0;
    turtle->state.step.vals[1] = 0.0;
    $(turtle, turn)(((double)iRun) * 2.0 * M_PI / 3.0);
    $(turtle, resize)(l);
    turtle->color = colors[iRun];
    double const ratioShrink = 1215.0 / 3333.0;
    loop(i, curve->len) {
      switch(curve->state[i].val) {
        case 'A':
          $(turtle, forward)(1);
          break;
        case 'B':
          $(turtle, forward)(1);
          break;
        case '+':
          $(turtle, turn)(M_PI / 3.0);
          break;
        case '-':
          $(turtle, turn)(-M_PI / 3.0);
          break;
        case 's':
          $(turtle, resize)(ratioShrink);
          break;
        case 'S':
          $(turtle, resize)(1.0 / ratioShrink);
          break;
        default:
          break;
      }
    }
  }
  $(img, saveToPath)("UnitTests/TestTurtleGraphic/gosper.png");
  CapyImg* checkImg = CapyImgLoadFromPath(
    "UnitTests/TestTurtleGraphic/checkGosper.png");
  CUTEST_ASSERT($(img, isSame)(checkImg), "turtle on Gosper failed");
  CapyTurtleGraphicFree(&turtle);
  CapyLSysGosperCurveFree(&curve);
  CapyImgFree(&img);
  CapyImgFree(&checkImg);
}
